// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview The recently closed menu: button, model data, and menu.
 */

cr.define('ntp', function() {
  'use strict';

  /**
   * Returns the text used for a recently closed window.
   * @param {number} numTabs Number of tabs in the window.
   * @return {string} The text to use.
   */
  function formatTabsText(numTabs) {
    if (numTabs == 1)
      return loadTimeData.getString('closedwindowsingle');
    return loadTimeData.getStringF('closedwindowmultiple', numTabs);
  }

  var Menu = cr.ui.Menu;
  var MenuItem = cr.ui.MenuItem;
  var MenuButton = cr.ui.MenuButton;
  var RecentMenuButton = cr.ui.define('button');

  RecentMenuButton.prototype = {
    __proto__: MenuButton.prototype,

    decorate: function() {
      MenuButton.prototype.decorate.call(this);
      this.menu = new Menu;
      cr.ui.decorate(this.menu, Menu);
      this.menu.classList.add('recent-menu');
      document.body.appendChild(this.menu);

      this.needsRebuild_ = true;
      this.hidden = true;
      this.anchorType = cr.ui.AnchorType.ABOVE;
      this.invertLeftRight = true;
    },

    addCommandItem: function(){
      var line = this.ownerDocument.createElement('i');
      line.className = 'recent-menu-line';
      this.menu.appendChild(line);
      cr.ui.decorate(line, MenuItem);
      var direction = 'ltr';
      if(this.dataItems_.length>0 && this.dataItems_[0].title != loadTimeData.getString('RecentMenuEmpty')){
        direction = this.dataItems_[0].direction;
        var cleanrestore_click_function = function(){
          chrome.send('cleanrestore_list');
          MenuButton.prototype.hideMenu.call(this);
        };
        this.addItem_({'direction':direction, 'title': loadTimeData.getString('cleanrestore_list'), 'type': 'tab', 'url':'chrome://last', 'click_function': cleanrestore_click_function});
      }
      var history_click_function = function(){
        chrome.send('navigateToUrl', ['chrome://history', '_blank', 0, false, false, false, false]);
        MenuButton.prototype.hideMenu.call(this);
      };
      this.addItem_({'direction':direction, 'title': loadTimeData.getString('show_full_history'), 'type': 'tab', 'url':'chrome://history', 'click_function': history_click_function});
    },

    /**
     * Shows the menu, first rebuilding it if necessary.
     * TODO(estade): the right of the menu should align with the right of the
     * button.
     * @override
     */
    showMenu: function() {
      if (this.needsRebuild_) {
        this.menu.textContent = '';
        this.dataItems_.forEach(this.addItem_, this);
        this.addCommandItem();
        this.needsRebuild_ = false;
      }

      MenuButton.prototype.showMenu.call(this);
    },

    /**
     * Sets the menu model data.
     * @param {Array} dataItems Array of objects that describe the apps.
     */
    set dataItems(dataItems) {
      // 始终显示
      if (dataItems.length == 0) {
        dataItems = [{title:loadTimeData.getString('RecentMenuEmpty')}];
      }
      this.dataItems_ = dataItems;
      this.needsRebuild_ = true;
      this.hidden = dataItems.length == 0;
    },

    /**
     * Adds an app to the menu.
     * @param {Object} data An object encapsulating all data about the app.
     * @private
     */
    addItem_: function(data) {
      var isWindow = data.type == 'window';
      var a = this.ownerDocument.createElement('a');
      a.className = 'recent-menu-item';
      if (isWindow) {
        a.href = '';
        a.classList.add('recent-window');
        a.textContent = formatTabsText(data.tabs.length);
      } else {
        if (data.url) {
          a.href = data.url;
          a.style.backgroundImage = 'url(chrome://favicon/' + data.url + ')';
        } else {
			a.style.paddingLeft = '8px';
        }
        a.textContent = data.title;
        // TODO(estade): add app ping url.
      }
      if(typeof(data.click_function)!='undefined'){
        a.onclick = function(){return false};
      }

      function onClick(e) {
        if(typeof(data.click_function)!='undefined'){
          data.click_function();
        }else{
        chrome.send('recordAppLaunchByURL',
                    [encodeURIComponent(data.url),
                     ntp.APP_LAUNCH.NTP_RECENTLY_CLOSED]);
        var index = Array.prototype.indexOf.call(a.parentNode.children, a);
        chrome.send('reopenTab', [data.sessionId, index,
            e.button, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey]);
        // We are likely deleted by this point!
        }
        e.preventDefault();
      }
      if (data.url || data.title != loadTimeData.getString('RecentMenuEmpty')) {
        a.addEventListener('click', onClick);
      }

      this.menu.appendChild(a);
      cr.ui.decorate(a, MenuItem);
    },
  };

  return {
    RecentMenuButton: RecentMenuButton,
  };
});
