// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * The global object.
 * @type {!Object}
 */
global = this;

/**
 * Alias for document.getElementById.
 * @param {string} id The ID of the element to find.
 * @return {HTMLElement} The found element or null if not found.
 */
function $(id) {
  return document.getElementById(id);
}
function $$(selector) {
    return document.querySelector(selector);
}
function $$$(selector) {
    return document.querySelectorAll(selector);
}

/**
 * Calls chrome.send with a callback and restores the original afterwards.
 * @param {string} name The name of the message to send.
 * @param {!Array} params The parameters to send.
 * @param {string} callbackName The name of the function that the backend calls.
 * @param {!Function} The function to call.
 */
function chromeSend(name, params, callbackName, callback) {
  var old = global[callbackName];
  global[callbackName] = function() {
    // restore
    global[callbackName] = old;

    var args = Array.prototype.slice.call(arguments);
    return callback.apply(global, args);
  };
  chrome.send(name, params);
}

/**
 * Generates a CSS url string.
 * @param {string} s The URL to generate the CSS url for.
 * @return {string} The CSS url string.
 */
function url(s) {
  // http://www.w3.org/TR/css3-values/#uris
  // Parentheses, commas, whitespace characters, single quotes (') and double
  // quotes (") appearing in a URI must be escaped with a backslash
  var s2 = s.replace(/(\(|\)|\,|\s|\'|\"|\\)/g, '\\$1');
  // WebKit has a bug when it comes to URLs that end with \
  // https://bugs.webkit.org/show_bug.cgi?id=28885
  if (/\\\\$/.test(s2)) {
    // Add a space to work around the WebKit bug.
    s2 += ' ';
  }
  return 'url("' + s2 + '")';
}

/**
 * Parses query parameters from Location.
 * @param {string} s The URL to generate the CSS url for.
 * @return {object} Dictionary containing name value pairs for URL
 */
function parseQueryParams(location) {
  var params = {};
  var query = unescape(location.search.substring(1));
  var vars = query.split("&");
  for (var i=0; i < vars.length; i++) {
    var pair = vars[i].split("=");
    params[pair[0]] = pair[1];
  }
  return params;
}

function findAncestorByClass(el, className) {
  return findAncestor(el, function(el) {
    if (el.classList)
      return el.classList.contains(className);
    return null;
  });
}

/**
 * Return the first ancestor for which the {@code predicate} returns true.
 * @param {Node} node The node to check.
 * @param {function(Node) : boolean} predicate The function that tests the
 *     nodes.
 * @return {Node} The found ancestor or null if not found.
 */
function findAncestor(node, predicate) {
  var last = false;
  while (node != null && !(last = predicate(node))) {
    node = node.parentNode;
  }
  return last ? node : null;
}

function swapDomNodes(a, b) {
  var afterA = a.nextSibling;
  if (afterA == b) {
    swapDomNodes(b, a);
    return;
  }
  var aParent = a.parentNode;
  b.parentNode.replaceChild(a, b);
  aParent.insertBefore(b, afterA);
}

/**
 * Disables text selection and dragging.
 */
function disableTextSelectAndDrag() {
  // Disable text selection.
  document.onselectstart = function(e) {
    e.preventDefault();
  }

  // Disable dragging.
  document.ondragstart = function(e) {
    e.preventDefault();
  }
}

/**
 * Check the directionality of the page.
 * @return {boolean} True if Chrome is running an RTL UI.
 */
function isRTL() {
  return document.documentElement.dir == 'rtl';
}

/**
 * Simple common assertion API
 * @param {*} condition The condition to test.  Note that this may be used to
 *     test whether a value is defined or not, and we don't want to force a
 *     cast to Boolean.
 * @param {string=} opt_message A message to use in any error.
 */
function assert(condition, opt_message) {
  'use strict';
  if (!condition) {
    var msg = 'Assertion failed';
    if (opt_message)
      msg = msg + ': ' + opt_message;
    throw new Error(msg);
  }
}

/**
 * Get an element that's known to exist by its ID. We use this instead of just
 * calling getElementById and not checking the result because this lets us
 * satisfy the JSCompiler type system.
 * @param {string} id The identifier name.
 * @return {!Element} the Element.
 */
function getRequiredElement(id) {
  var element = $(id);
  assert(element, 'Missing required element: ' + id);
  return element;
}

// Handle click on a link. If the link points to a chrome: or file: url, then
// call into the browser to do the navigation.
document.addEventListener('click', function(e) {
  // Allow preventDefault to work.
  if (!e.returnValue)
    return;

  var el = e.target;
  if (el.nodeType == Node.ELEMENT_NODE &&
      el.webkitMatchesSelector('A, A *')) {
    while (el.tagName != 'A') {
      el = el.parentElement;
    }

    if ((el.protocol == 'file:' || el.protocol == 'about:') &&
        (e.button == 0 || e.button == 1)) {
      chrome.send('navigateToUrl', [
        el.href,
        el.target,
        e.button,
        e.altKey,
        e.ctrlKey,
        e.metaKey,
        e.shiftKey
      ]);
      e.preventDefault();
    }
  }
});

/**
 * Creates a new URL which is the old URL with a GET param of key=value.
 * @param {string} url The base URL. There is not sanity checking on the URL so
 *     it must be passed in a proper format.
 * @param {string} key The key of the param.
 * @param {string} value The value of the param.
 * @return {string}
 */
function appendParam(url, key, value) {
  var param = encodeURIComponent(key) + '=' + encodeURIComponent(value);

  if (url.indexOf('?') == -1)
    return url + '?' + param;
  return url + '&' + param;
}

/**
 * @method domReady 
 * @param {function} fn
 *
 * add by lichao3@360.cn
 */
function domReady(fn) {
    /in/.test(document.readyState) ? setTimeout(domReady, 0, fn) : fn();
}

/**
 * @method encodeURIJson 
 * @param {Json} json  
 *
 * @returns {string} : 
 */
function encodeURIJson(json) {
    var s = [];
    for (var p in json) {
        if (json[p] == null) continue;
        if (json[p] instanceof Array) {
            for (var i = 0; i < json[p].length; i++) s.push(encodeURIComponent(p) + '=' + encodeURIComponent(json[p][i]));
        }
        else s.push(encodeURIComponent(p) + '=' + encodeURIComponent(json[p]));
    }
    return s.join('&');
}

/**
 * @method xhr
 * @param {object} opts
 *
 * add by lichao3@360.cn
 */
function xhr(opts) {

    const STATE_INIT = 0,
    STATE_REQUEST = 1,
    STATE_SUCCESS = 2,
    STATE_ERROR = 3,
    STATE_TIMEOUT = 4,
    STATE_CANCEL = 5;

    var defaultRequestHeaders = {
        'Content-type': 'application/x-www-form-urlencoded; charset=UTF-8' //from jq-1.72
    };

    var method = opts.method || 'get',
    url = opts.url || '',
    data = opts.data || '',
    async = 'async' in opts ? opts.async: true,
    user = opts.user || '',
    pwd = opts.pwd || '',
    oncomplete = opts.oncomplete,
    onsuccess = opts.onsuccess,
    onerror = opts.onerror,
    oncancel = opts.oncancel,
    ontimeout = opts.ontimeout,
    timeout = opts.timeout || 30000;

    var oXhr = new XMLHttpRequest(),
    state = STATE_INIT;

    if (typeof data == 'object') {
        data = encodeURIJson(data);
    }

    if (data && method != 'post') {
        url += (url.indexOf('?') != - 1 ? '&': '?') + data;
    }
    if (user) {
        oXhr.open(method, url, async, user, pwd);
    } else {
        oXhr.open(method, url, async);
    }
    for (var i in defaultRequestHeaders) {
        oXhr.setRequestHeader(i, defaultRequestHeaders[i]);
    }

    var timer;

    if (async) {
        oXhr.onreadystatechange = function() {
            if (oXhr.readyState == 4) {
                if (state != STATE_TIMEOUT && state != STATE_CANCEL) {
                    var status = oXhr.status;
                    if ((status >= 200 && status < 300) || status == 304) {
                        state = STATE_SUCCESS;
                        onsuccess && onsuccess(xhr);
                    } else {
                        state = STATE_ERROR;
                        onerror && onerror(xhr);
                    }
                    oncomplete && oncomplete(xhr);
                }
            }
        };
        timer = setTimeout(function() {
            state = STATE_TIMEOUT;
            oXhr.abort();
        },
        timeout);
    }

    if (method == 'post') {
        oXhr.send(data || ' ');
    } else {
        oXhr.send(null);
    }
    state = STATE_REQUEST;

    return {
        isProcessing: function() {
            return oXhr.readyState > 0 && oXhr.readyState < 4;
        },
        cancel: function() {
            if (this.isProcessing()) {
                state = STATE_CANCEL;
                oXhr.abort();
                return true;
            }
            return false;
        }
    };
}

/**
 * @method jsonp
 *
 * add by lichao3@360.cn
 */
function jsonp(opts) {

    var url = opts.url || '',
    callbackName = opts.callbackName || 'jsonp_callback',
    callbackKey = opts.callbackKey || '_jsonp',
    onsuccess = opts.onsuccess,
    onerror = opts.onerror;

    var isCalled = false;

    var hookBackup = window[callbackName],
    callback = window[callbackName] = function() {
        window[callbackName] = hookBackup;
        isCalled = true;
        onsuccess && onsuccess.apply(null, arguments);
    };

    var sct = document.createElement('script');

    sct.onload = function() {
        sct.parentNode.removeChild(sct);
        if (!isCalled) {
            window[callbackName] = hookBackup;
            onerror && onerror(); 
        }
    };
    url += (url.indexOf('?') != - 1 ? '&': '?') + callbackKey + '=' + callbackName;
    sct.src = url;

    document.body.appendChild(sct);

}
// 该函数把特殊符号的对象entityMap依次遍历并且存在一个数组里面
var keys = Object.keys || function (obj) {
  obj = Object(obj)
  var arr = []
  for (var a in obj) arr.push(a)
  return arr
}
// 该函数是把特殊符号和转义后的字符进行反转。
// 比如说原来是'＆': '＆amp;'对应，现在是'＆amp;': '＆'对应
var invert = function (obj) {
  obj = Object(obj)
  var result = {}
  for (var a in obj) result[obj[a]] = a
  return result
}
// 把特殊符号和转移后的字符一一对应起来并且存在对象里面
var entityMap = {
  escape: {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      ' ': '&nbsp;',
  }
}
// 把特殊符号和转义后的字符进行反转后以unescape作为关键字存在entityMap中
entityMap.unescape = invert(entityMap.escape);

var entityReg = {
  escape: RegExp('[' + keys(entityMap.escape).join('') + ']', 'g'),
  unescape: RegExp('(' + keys(entityMap.unescape).join('|') + ')', 'g')
}
// 将HTML转义为实体
function HTMLescape(html) {
  if (typeof html != 'string') return ''
  return html.replace(entityReg.escape, function (match) {
      return entityMap.escape[match]
  })
}

/**
 * @method tmpl
 *
 * @ from qwrap
 */
var tmpl= (function() {
    
    var sArrName = "sArrCMX",
        sLeft = sArrName + '.push("';
   
    var tags = {
        'js': {
            tagG: 'js',
            isBgn: 1,
            isEnd: 1,
            sBgn: '");',
            sEnd: ';' + sLeft
        },

        'if': {
            tagG: 'if',
            isBgn: 1,
            rlt: 1,
            sBgn: '");if',
            sEnd: '{' + sLeft
        },

        'elseif': {
            tagG: 'if',
            cond: 1,
            rlt: 1,
            sBgn: '");} else if',
            sEnd: '{' + sLeft
        },

        'else': {
            tagG: 'if',
            cond: 1,
            rlt: 2,
            sEnd: '");}else{' + sLeft
        },

        '/if': {
            tagG: 'if',
            isEnd: 1,
            sEnd: '");}' + sLeft
        },

        'for': {
            tagG: 'for',
            isBgn: 1,
            rlt: 1,
            sBgn: '");for',
            sEnd: '{' + sLeft
        },

        '/for': {
            tagG: 'for',
            isEnd: 1,
            sEnd: '");}' + sLeft
        },

        'while': {
            tagG: 'while',
            isBgn: 1,
            rlt: 1,
            sBgn: '");while',
            sEnd: '{' + sLeft
        },

        '/while': {
            tagG: 'while',
            isEnd: 1,
            sEnd: '");}' + sLeft
        }
    };

    return function(sTmpl, opts) {
        var N = -1,
            NStat = [];
        var ss = [
            [/\{strip\}([\s\S]*?)\{\/strip\}/g, function(a, b) {
                return b.replace(/[\r\n]\s*\}/g, " }").replace(/[\r\n]\s*/g, "");
            }],
            [/\\/g, '\\\\'],
            [/"/g, '\\"'],
            [/\r/g, '\\r'],
            [/\n/g, '\\n'],
            [
                /\{[\s\S]*?\S\}/g,
                function(a) {
                    a = a.substr(1, a.length - 2);
                    for (var i = 0; i < ss2.length; i++) {a = a.replace(ss2[i][0], ss2[i][1]); }
                    var tagName = a;
                    if (/^(.\w+)\W/.test(tagName)) {tagName = RegExp.$1; }
                    var tag = tags[tagName];
                    if (tag) {
                        if (tag.isBgn) {
                            var stat = NStat[++N] = {
                                tagG: tag.tagG,
                                rlt: tag.rlt
                            };
                        }
                        if (tag.isEnd) {
                            if (N < 0) {throw new Error("Unexpected Tag: " + a); }
                            stat = NStat[N--];
                            if (stat.tagG != tag.tagG) {throw new Error("Unmatch Tags: " + stat.tagG + "--" + tagName); }
                        } else if (!tag.isBgn) {
                            if (N < 0) {throw new Error("Unexpected Tag:" + a); }
                            stat = NStat[N];
                            if (stat.tagG != tag.tagG) {throw new Error("Unmatch Tags: " + stat.tagG + "--" + tagName); }
                            if (tag.cond && !(tag.cond & stat.rlt)) {throw new Error("Unexpected Tag: " + tagName); }
                            stat.rlt = tag.rlt;
                        }
                        return (tag.sBgn || '') + a.substr(tagName.length) + (tag.sEnd || '');
                    } else {
                        return '",(' + a + '),"';
                    }
                }
            ]
        ];
        var ss2 = [
            [/\\n/g, '\n'],
            [/\\r/g, '\r'],
            [/\\"/g, '"'],
            [/\\\\/g, '\\'],
            [/\$(\w+)/g, 'opts["$1"]'],
            [/print\(/g, sArrName + '.push(']
        ];
        for (var i = 0; i < ss.length; i++) {
            sTmpl = sTmpl.replace(ss[i][0], ss[i][1]);
        }
        if (N >= 0) {throw new Error("Lose end Tag: " + NStat[N].tagG); }
        sTmpl = 'var ' + sArrName + '=[];' + sLeft + sTmpl + '");return ' + sArrName + '.join("");';

        var fun = new Function('opts', sTmpl);
        if (arguments.length > 1) {return fun(opts); }
        return fun;
    };
}());
